/* sccsid[] = "@(#) $Id: //bas/46C/src/proj/crfc/crfcsapp.cpp#2 $ SAP" */   
//
// SAP RFC Classes C++ library.
// Copyright (C) 1996 SAP America, Inc.
// All rights reserved. 

//////////////////////////////////////////////////////////////////////////////////
//  File crfcsapp.cpp
//////////////////////////////////////////////////////////////////////////////////
#include <ctype.h>
#include <stdio.h>
#include "sapitab.h"
#include "saprfc.h"

#include "crfcserv.h"
#include "crfcconn.h"
#include "crfcsapp.h"
                                                                
extern CRfcTrace RfcClassTrace;

#if defined(CRFCwithTHREADS)
CSAPCriticalSection serverAppLock;
#endif

BOOL CRfcServerApp::IsMultiThreading = FALSE;

#if defined(CRFCwithTHREADS)
int rfc_thread[10000];
#endif

//Initialize RFC Server application instance number 
unsigned CRfcServerApp::m_nAppCreated = 0 ;

//RFC lib desiganated user RFC dispatcher function name
RFC_FUNCTIONNAME CRfcServerApp::m_strDispatcher = "%%USER_GLOBAL_SERVER" ;

CRfcServerApp*   CRfcServerApp::m_pInstance = NULL ;

#define SAP_RFC_GETNAME

CRfcServerApp::CRfcServerApp (CRfcConnection* pConnection)
{
#if defined(CRFCwithTHREADS)
	serverAppLock.Lock();
#endif
    m_nAppCreated++ ;     //Increse instance count

    if (m_nAppCreated == 1)
    {
       m_pConnection = pConnection ;

       //Install the ServerApp's dispatcher
#ifdef _WIN32
       InstallFunction(m_strDispatcher, 
		       (RFC_ONCALL) Dispatch,  //Dispatcher function
                       DispatchDocu());       //returns description for the dispatcher 
#elif defined(SAPonOS390)
       InstallFunction(m_strDispatcher, 
		       (RFC_ONCALL) CRfcServerApp::Dispatch,  //Dispatcher function
                       DispatchDocu());       //returns description for the dispatcher 
#else
       InstallFunction(m_strDispatcher, 
		       CRfcServerApp::Dispatch,  //Dispatcher function
                       DispatchDocu());       //returns description for the dispatcher 
#endif

       //Only one instance of this class can be created, 
       //so remember this instance                        
        m_pInstance = this;
    }
    else
    {
        RfcClassTrace.TraceTime();
        RfcClassTrace.Trace(CRFCSAPP_CONSTRUCTOR);
        RfcClassTrace.Trace(ALLOW_ONE_INSTANCE);
#if defined(CRFCwithTHREADS)
		serverAppLock.Unlock();
#endif
        throw ALLOW_ONE_INSTANCE ;
    }
#if defined(CRFCwithTHREADS)
	serverAppLock.Unlock();
#endif
}


CRfcServerApp::~CRfcServerApp (void)
{
#if defined(CRFCwithTHREADS)
	serverAppLock.Lock();
#endif
   m_nAppCreated-- ;
#if defined(CRFCwithTHREADS)
	serverAppLock.Unlock();
#endif
}



void CRfcServerApp::SetConnection (CRfcConnection& connection)
{ 
#if defined(CRFCwithTHREADS)
	serverAppLock.Lock();
#endif
    m_pConnection = &connection; 
#if defined(CRFCwithTHREADS)
	serverAppLock.Unlock();
#endif
}


void CRfcServerApp::AddServerFunction(CRfcServerFunc& serverFunc)
{ 
    //Only add description to the the library function list, 
    //the real function will be handled by the ServerApp
    ::RfcInstallFunction ((char*) serverFunc.GetFunctionName(), NULL, 
                          (char*) serverFunc.GetDescription());

#if defined(CRFCwithTHREADS)
	serverAppLock.Lock();
#endif
	try
	{
		m_FunctionList.AppendElement (serverFunc); 
		serverFunc.SetServerApp(this);
	} 
	catch(...)
	{
#if defined(CRFCwithTHREADS)
		serverAppLock.Unlock();
#endif
		throw;
	}
#if defined(CRFCwithTHREADS)
	serverAppLock.Unlock();
#endif
}

void CRfcServerApp::RemoveServerFunction (int nIndex )
{ 
	if(m_FunctionList.GetSize() <= 0) return;
#if defined(CRFCwithTHREADS)
	serverAppLock.Lock();
#endif
	CRfcServerFunc * servFunc = GetServerFunction(nIndex);
	if (!servFunc)
	{
#if defined(CRFCwithTHREADS)
		serverAppLock.Unlock();
#endif
		return;
	}
	try
	{
		RFC_FUNCTIONNAME funcName;
		strcpy(funcName, servFunc->GetFunctionName());
		::RfcUninstallFunction(funcName);
		m_FunctionList.RemoveElement (nIndex); 
	}
	catch(...)
	{
#if defined(CRFCwithTHREADS)
		serverAppLock.Unlock();
#endif
		throw;
	}
#if defined(CRFCwithTHREADS)
	serverAppLock.Unlock();
#endif
}

void CRfcServerApp::RemoveServerFunction (RFC_FUNCTIONNAME functionName )
{ 
	if(m_FunctionList.GetSize() <= 0) return;
#if defined(CRFCwithTHREADS)
	serverAppLock.Lock();
#endif
	try
	{
		::RfcUninstallFunction(functionName);
		m_FunctionList.RemoveElement (CSTR(functionName)); 
	}
	catch(...)
	{
#if defined(CRFCwithTHREADS)
		serverAppLock.Unlock();
#endif
		throw;
	}
#if defined(CRFCwithTHREADS)
	serverAppLock.Unlock();
#endif
}


                             
//Main loop of the server app, which takes care of listening to  
//incoming requests, dispatching to appropriate server function
//objects, and calling ReceiveData/Process/ReturnCall them.
//May throw CRFC_ERROR_INFO exception when getting RFC 
//name and data, sending RFC data.
#if defined(CRFCwithTHREADS) 
RFC_RC   CRfcServerApp::Run(int nChildThreads)
#else
RFC_RC   CRfcServerApp::Run(void)
#endif
{
#if defined(CRFCwithTHREADS) 
	if (nChildThreads > 0)
	{ 
		int nThreads = 0;
		IsMultiThreading = TRUE;
		try
		{
			nThreads = StartThreadServer(nChildThreads);
		} catch (CRFC_ERROR_INFO & rfcError) 
		{
			throw rfcError;
			return RFC_FAILURE;
		} catch (char * pException)
		{
			throw pException;
			return RFC_EXCEPTION;
		}
		
		if (nThreads != nChildThreads)
		{
			int nFail =  nChildThreads - nThreads;
			char myMsg[128];
			memset(myMsg, 0, 127);
			sprintf(myMsg, "%4d threads failed in this server.\n", nFail);
			RfcClassTrace.TraceTime();
            RfcClassTrace.Trace(CRFCSAPP_RUN);
 			RfcClassTrace.Trace(myMsg);
			throw CRFCSAPP_RUN_THREADS_ERROR;
		}

		return RFC_OK;
	}
#endif

    RFC_RC           rc ;
	RFC_HANDLE RfcHandle = GetConnectionRfcHandle();
	if (RfcHandle == RFC_HANDLE_NULL)
		throw INVALID_RFC_HANDLE;

    do 
    {  
        for ( rc = RFC_RETRY; rc == RFC_RETRY;)
        {
            // Listen for pending RFC activities
            rc = ::RfcListen (RfcHandle) ;
           
            if (rc == RFC_RETRY)
            {
                OnIdle() ;
            }
		}

      //  _asm int 3 ; 
//
        if (rc == RFC_CLOSED)
        {
            break ;  // Usually server task completed
        }
        else if (rc != RFC_OK) 
        {
            RfcClassTrace.TraceTime();
            RfcClassTrace.Trace(CRFCSAPP_RUN);
            RfcLastErrorEx (&m_RfcErrorInfo) ;
			RfcLastError (&m_RfcErrorInfo) ;
            RfcClassTrace.TraceRFCErrorInfo((RFC_ERROR_INFO_EX) m_RfcErrorInfo);
            throw m_RfcErrorInfo ;
        } 

		rc = ::RfcDispatch (RfcHandle) ;

        if (rc != RFC_OK && rc != RFC_CLOSED)
        {
            RfcClassTrace.TraceTime();
            RfcClassTrace.Trace(CRFCSAPP_RUN);
            RfcLastErrorEx (&m_RfcErrorInfo) ;
			RfcLastError (&m_RfcErrorInfo) ;
            RfcClassTrace.TraceRFCErrorInfo((RFC_ERROR_INFO_EX) m_RfcErrorInfo);
            throw m_RfcErrorInfo ;
        } 

    }
    while (rc == RFC_OK) ;

    return rc ;
}
 

//Stop the ServerApp. The default implementation closes the 
//RFC connection so that it will result stopping the run loop. 
//Users can provide their own means of ServerApp stop implementation.
void CRfcServerApp::ExitInstance (void)
{
    m_pConnection->Close() ;
}

//Do nothing for the default implementation
//However, it can be overwritten
void CRfcServerApp::OnIdle()
{
}

//Look up function name and invoke the serverFunc
void CRfcServerApp::Invoke (RFC_HANDLE RfcHandle, 
                           RFC_FUNCTIONNAME FunctionName)                                
{

	CRfcServerFunc*  pRfcFunction = NULL ;

#if defined(CRFCwithTHREADS)
	if (IsMultiThreading)
	{
		serverAppLock.Lock();

		if (strcmp(FunctionName, "RFCPING") == 0) 
		{
			pRfcFunction = new CRfcPing(FunctionName);
			if (pRfcFunction == NULL)
			{
				RfcClassTrace.TraceTime();
				RfcClassTrace.Trace(CRFCSAPP_INVOKE);
				RfcClassTrace.Trace(MEMORY);
				serverAppLock.Unlock();
				throw MEMORY;
			}
		}
		else
		{
			CRfcServerFunc * pTempSvrApp = m_FunctionList[FunctionName];
			if (pTempSvrApp != NULL)
			{
				try
				{
					pRfcFunction = pTempSvrApp->clone();
				} catch(char * pException)
				{
					serverAppLock.Unlock();
					throw pException;
				}

 				if (pRfcFunction == NULL)
				{
					RfcClassTrace.TraceTime();
					RfcClassTrace.Trace(CRFCSAPP_INVOKE);
					RfcClassTrace.Trace(CRFCSAPP_INVOKE_CLONE_ERROR);
					serverAppLock.Unlock();
					throw CRFCSAPP_INVOKE_CLONE_ERROR;
				}
			}
			else
				pRfcFunction = NULL;
		}
		
		if (pRfcFunction == NULL)
		{

			::RfcRaise (RfcHandle, FUNCTION_NOT_FOUND); 
			RfcClassTrace.TraceTime();
			RfcClassTrace.Trace(CRFCSAPP_INVOKE);
			RfcClassTrace.Trace(FUNCTION_NOT_FOUND);
			serverAppLock.Unlock();
			throw (FUNCTION_NOT_FOUND) ;
		}

		CRfcConnection   Connection ;
		try
		{
			Connection.FromHandle(RfcHandle,FALSE) ;
			pRfcFunction->SetConnection (Connection);
		} 
		catch (char * pException)
		{
			throw pException;
		}
		catch (RFC_ERROR_INFO & err)
		{
			throw err;
		}
		serverAppLock.Unlock();

		pRfcFunction->MultipleThreadDataProcess();
			// pRfcFunction will be self destroyed in this method

	}
	else  //  No multithreading
#endif
	{

		// _asm int 3;
		//Retrieve the server Function object from the list
		if (strcmp(FunctionName, "RFCPING") == 0) 
		{
			pRfcFunction = new CRfcPing(FunctionName);
			if (pRfcFunction == NULL)
			{
				RfcClassTrace.TraceTime();
				RfcClassTrace.Trace(CRFCSAPP_INVOKE);
				RfcClassTrace.Trace(MEMORY);
				throw MEMORY;
			}
		}
		else
		{
			pRfcFunction = m_FunctionList[FunctionName];
			if (pRfcFunction == NULL)
			{

				::RfcRaise (RfcHandle, FUNCTION_NOT_FOUND); 
				RfcClassTrace.TraceTime();
				RfcClassTrace.Trace(CRFCSAPP_INVOKE);
				RfcClassTrace.Trace(FUNCTION_NOT_FOUND);
				throw (FUNCTION_NOT_FOUND) ;
			}
		}	// end if (strcmp(FunctionName, "RFCPING") == 0)
	
		//Use incoming handle to create a temp connection object. 
		//Don't use the connection object in serverApp, it does
		//not work if there is a transactional RFC call.


		try
		{
			CRfcConnection   Connection ;
			Connection.FromHandle(RfcHandle,FALSE) ;
			pRfcFunction->SetConnection (Connection);

			pRfcFunction->ReceiveData ();

			// At this point, the server app can call back to R/3 to get
			//  R/3 system parameter (most importantly, the R/3 release version
			//  information).  However, the RfcHandle passed into this function
			//  cannot be used if we are actually accepting a transactional
			//  RFC call since the RFC handle for transactional RFC calls
			//  change.  The RFC handle used for callback must be the original
			//  RFC handle obtained from the first CRfcConnection::Accept() call.
			// That original RFC handle is still stored in the CRfcConnection
			//  object that made the Accept() call and that CRfcConnection object
			//  is pointed to by this->m_pConnection.  So, to do the callback
			//  for R/3 system parameters, all we have to do is use the original
			//  CRfcConnection object that made the Accept() call, and have
			//  that object call GetR3Release().
			//  In multithreading server, each thread has its only connectio
			//  with a different RfcHandle, Connection should be used here instead
			//  of m_pConnection.
			if((m_pConnection != NULL) &&
				(m_pConnection->GetConnectInfo()->rstrR3release == RELEASEUNKNOWN))
			{
				m_pConnection->GetR3Release();
				if(m_pConnection->GetConnectInfo()->bGetR3SystemInfo)
					m_pConnection->GetR3SystemInfo();
			}

			pRfcFunction->Process ();
			pRfcFunction->ReturnCall ();
		}
		catch (CRFC_ERROR_INFO) 
		{
			if (strcmp(FunctionName, "RFCPING") == 0) 
			{
					if (pRfcFunction != NULL)
					{
						delete pRfcFunction;
						pRfcFunction = NULL;
					}
			}

//Does not handle exception here, pass to caller
			throw; 
		}  //  End of exceptional handling.


		if (strcmp(FunctionName, "RFCPING") == 0) 
		{
				if (pRfcFunction != NULL)
				{
					delete pRfcFunction;
					pRfcFunction = NULL;
				}
		}
	}  // End of No multithreading or end of if(IsMultiThreading)
}



RFC_RC DLL_CALL_BACK_FUNCTION CRfcServerApp::Dispatch (RFC_HANDLE RfcHandle) 
{
	 RFC_FUNCTIONNAME FunctionName ;
     RFC_RC           rc ;

     RfcClassTrace.Trace(CRFCSAPP_DISPATCH);

#if defined(CRFCwithTHREADS)
		serverAppLock.Lock();
#endif

     rc = ::RfcGetName(RfcHandle, FunctionName) ;
     
     if (rc != RFC_OK) 
	 {
#if defined(CRFCwithTHREADS)
		 serverAppLock.Unlock();
#endif
		 return rc ;
	 }

#if defined(CRFCwithTHREADS)
		 serverAppLock.Unlock();
#endif

	 try
	 {    
		 m_pInstance->Invoke (RfcHandle, FunctionName) ;
	 } catch (char *pException)
	 {
#if defined(CRFCwithTHREADS)
			serverAppLock.Lock();
#endif

		 printf("Exception in CRfcServerApp::Dispatch : %s\n", pException);

#if defined(CRFCwithTHREADS)
			PRINT_RFC_CALL(rfc_thread[RfcHandle], pException);
			serverAppLock.Unlock();
#endif
	 } catch (RFC_ERROR_INFO_EX &error_info)
	 {
		 char               ebuf[1024];

#if defined(CRFCwithTHREADS)
			serverAppLock.Lock();
#endif

		sprintf(ebuf, "\nRFC call/exception during Invoke of CRfcServerApp\n");
		sprintf(ebuf+strlen(ebuf), "Key         %s\n", error_info.key);
		sprintf(ebuf+strlen(ebuf), "Message     %s\n", error_info.message);
		printf("%s", ebuf);
#if defined(CRFCwithTHREADS)
			PRINT_RFC_CALL(rfc_thread[RfcHandle], ebuf);
			serverAppLock.Unlock();
#endif
	 }


     return RFC_OK ;
}



char* CRfcServerApp::DispatchDocu (void)
{
    RfcClassTrace.Trace(CRFCSAPP_DISPATCHDOCU);

#if defined(CRFCwithTHREADS)
	serverAppLock.Lock();
#endif

    static char docu[] =
      "The RFC library will call this function if any unknown"            NL
      "RFC function should be executed in this RFC server program."       NL
      ;

#if defined(CRFCwithTHREADS)
	serverAppLock.Unlock();
#endif

    return docu;
}


RFC_RC CRfcServerApp::Dispatch (void)
{
	RFC_HANDLE rfcHandle = GetConnectionRfcHandle();
	if (rfcHandle == RFC_HANDLE_NULL) 
		throw INVALID_RFC_HANDLE; 
	return ::RfcDispatch (rfcHandle);
}



CRfcPing::CRfcPing (CSTR strFuncName)
: CRfcServerFunc(strFuncName)
{
}

CRfcPing::~CRfcPing ()
{
}

void CRfcPing::Process (void)
{
}

char * CRfcPing::GetDescription (void)
{
	 static char docu[] =
      "RFC Server: RFCRING program.";

     return docu;
}


#if defined(CRFCwithTHREADS)

#include <windows.h>
#if !defined(EBUSY) || !defined(ESRCH)
  #include <errno.h>
#endif

#if !defined(EBUSY) || !defined(ESRCH)
#define EBUSY 1
#define ESRCH -1
#endif

typedef HANDLE THREAD_ID_TYPE;

#define THREAD_INVALID_ID                    (THREAD_ID_TYPE)0xffffffff

/*--------------------------------------------------------------------*/
/* Output RFC-Call on screen                                */
/*--------------------------------------------------------------------*/
void CRfcServerApp::PRINT_RFC_CALL(int thread_id, char * text)
{                             
	if (thread_id)
		printf("\n Thread %3d:  %s", thread_id, text);              
	else                                                          
		printf("\n Main Thread  %s", text);                         
	fflush(stdout);
}

/*--------------------------------------------------------------------*/
/* Error Cleanup because of a RFC-Error                               */
/*--------------------------------------------------------------------*/
void CRfcServerApp::rfc_error(char *operation, int thread_id)
{
  char               ebuf[1024];
  RFC_ERROR_INFO_EX error_info;
  char temBuf[128];

  memset(ebuf, 0, sizeof(ebuf));
  sprintf(ebuf, "\n RfcLastErrorEx\n\n");

  RfcLastErrorEx(&error_info);

  ErrorGroupName(error_info.group, temBuf);

  sprintf(ebuf+strlen(ebuf), "\nRFC call/exception: %s\n", operation);
  sprintf(ebuf+strlen(ebuf), "Group       Error group %d\n", temBuf);
  sprintf(ebuf+strlen(ebuf), "Key         %s\n", error_info.key);
  sprintf(ebuf+strlen(ebuf), "Message     %s\n", error_info.message);
  PRINT_RFC_CALL(thread_id, ebuf);
  RfcClose(RFC_HANDLE_NULL);
  return;
}


/*--------------------------------------------------------------------*/
/* Issue RfcAbort with Abort Text because of an Application Error     */
/*--------------------------------------------------------------------*/
void CRfcServerApp::function_abort(RFC_HANDLE rfc_handle, char *atext)
{
	int           thread_id = rfc_thread[rfc_handle];
  char          dbuf[512];

  sprintf(dbuf, "RfcAbort         %s\n", atext);
  PRINT_RFC_CALL(thread_id, dbuf);

  RfcAbort(rfc_handle, atext);
  return;
}


#define SNC_NAME_LN            256
/*--------------------------------------------------------------------*/
/* SNC check: - Working with SNC                                      */
/*            - If yes, check SNC name or SNC ACL key                 */
/*                                                                    */
/* SNC check should be done                                           */
/*   - after an successful RfcGetName                                 */
/*   - after an successful RfcListen for waiting for RFC request      */
/*   - after an successful RfcWaitForRequest                          */
/*   - if working with RfcDispatch, in each RFC function and before   */
/*     RfcGetData                                                     */
/*--------------------------------------------------------------------*/
int CRfcServerApp::SNC_check(RFC_HANDLE rfc_handle)
{
  int              i,
		   aclkey_len;
  char             partner_name[SNC_NAME_LN+1];
  RFC_RC           rfc_rc;
  RFC_BYTE         partner_aclkey[SNC_NAME_LN+1];
  RFC_SNC_MODE     snc_mode;
  char             abort_text[80];
  char             dbuf[512];
  int              thread_id = rfc_thread[rfc_handle];

  rfc_rc = RfcSncMode(rfc_handle,
		      &snc_mode);


  if (rfc_rc != RFC_OK)
  {
    rfc_error("RfcSncMode", thread_id);
    return 99;
  }

  if (snc_mode == RFC_SNC_MODE_OFF)
    return 0;

  rfc_rc = RfcSncPartnerName(rfc_handle,
			     partner_name,
			     SNC_NAME_LN);

  if (rfc_rc != RFC_OK)
  {
    rfc_error("RfcSncPartnerName", thread_id);
    return 99;
  }

  sprintf(dbuf, "SNC-PartnerName  %s", partner_name);
  PRINT_RFC_CALL(thread_id, dbuf);

  memset(partner_aclkey, ' ', SNC_NAME_LN);
  rfc_rc = RfcSncPartnerAclKey(rfc_handle,
			       partner_aclkey,
			       SNC_NAME_LN,
			       &aclkey_len);

  if (rfc_rc != RFC_OK)
  {
    rfc_error("RfcSncPartnerAclKey", thread_id);
    return 99;
  }

  sprintf(dbuf, "PartnerAclKey    ");

  for (i=0; i<aclkey_len; i++)
    sprintf(dbuf+strlen(dbuf), "%02X", (int) partner_aclkey[i]);

  PRINT_RFC_CALL(thread_id, dbuf);

  /* Sample coding for check and refuse */
  if (strcmp(partner_name, "not_authorized_snc_name") == 0)
  {
    sprintf(abort_text,
	    "SNC name '%s' is not authorized for calling this program",
	    partner_name);
    function_abort(rfc_handle, abort_text);
    return 99;
  }
  return 0;
}

char ** new_argv;
char * codepage;
int globTraceLevel;

/*****************************************************************
  This member function is for starting a multithreading server
  with CRfc class library.  Users of this method need to give
  the number threads for the applicaiton.  When number of child 
  threads is zero, this function is equal to Run(void).
  Parameter:  Number of desired child threads.
  Return value:  Number of successful child threads generated.
******************************************************************/


int CRfcServerApp::StartThreadServer(int nChildThreads)
{
	const CRfcConnection * connection = GetConnection();
	if (connection == NULL)
	{	
		RfcClassTrace.TraceTime();
        RfcClassTrace.Trace(CRFCSAPP_STARTTHREADSERVER);
 		RfcClassTrace.Trace(CRFCSAPP_THREAD_NO_CONNECTION);
 		throw CRFCSAPP_THREAD_NO_CONNECTION;
	}

	new_argv = connection->GetCommandLineArgument();
	memset(rfc_thread, 0, 9999);
	globTraceLevel = connection->GetTraceLevel();
	codepage = connection->GetCodePage();

	if (nChildThreads > max_threads) 
		nChildThreads = max_threads;

	THREAD_ID_TYPE hThread[max_threads];
    THREAD_ID_TYPE dwThreadId;
    int            i,
		   thread_rc,
		   active = 0,
		   success = 0,
		   failure = 0;
	unsigned long dwExitCode = 0;

    memset(hThread, 0, sizeof(hThread));

	char dbuf[512];
	memset(dbuf, 0, 511);
	sprintf(dbuf, "StartThreadServer");
	PRINT_RFC_CALL(0, dbuf);

	for (i=1; i<=nChildThreads; i++)
	{
		hThread[i] = CreateThread(NULL, 0, (LPTHREAD_START_ROUTINE) CRfcServerApp::RFC_RegisterThread, 
			                      (LPVOID)((void *)i), 0, (LPDWORD) (&dwThreadId));
	}

    for(;;)
    {
      active = 0;
      for (i=1; i<=nChildThreads; i++)
      {
		if (hThread[i] != THREAD_INVALID_ID)
		{
			thread_rc = GetExitCodeThread(hThread[i], &dwExitCode) ?
				        ((dwExitCode == STILL_ACTIVE) ? EBUSY : 0) : ESRCH;
			if (thread_rc == EBUSY)
			{
				active = 1;
			}
			else
			{
				if (thread_rc == 0)
				{	
					if (dwExitCode == 0)
						success++;
					else
					{
						if (dwExitCode == 99)
							printf("\n\n Main Thread:  RFC-ERROR OCCURS  ==>  Thread %d exited\n", i);
						else
							printf("\n\n Main Thread:  Thread %d exited with rc=%d\n",
									i, dwExitCode);
						failure++;
					}
				}
				else
				{
					printf("\n\n Main Thread:  Thread %d returned %d\n",
						i, thread_rc);
					failure++;
				}
				CloseHandle(hThread[i]);
				hThread[i] = THREAD_INVALID_ID;
			}
		}
      }
      if (active == 0)
			break;

      /* Don't steal all the CPU */
#if defined(SAPonNT)
      Sleep(1000);
#elif defined(SAPonUNIX)
      sleep(1);
#endif
    }

    if (success)
      printf("\n\n Main Thread:  %d thread(s) successfully terminated.\n", success);

    if (failure)
      printf("\n\n Main Thread:  %d thread(s) terminated with an error.\n", failure);

	return success;
}

/*====================================================================*/
/*                                                                    */
/* Thread for working as RFC server							          */
/*                                                                    */
/*====================================================================*/


int CRfcServerApp::RFC_RegisterThread(void *lpParms)
{
  RFC_HANDLE        rfc_handle;
  RFC_RC            rfc_rc;
  char              dbuf[512];
  RFC_ERROR_INFO_EX error_info;

  for (;;)
  { 
    /*------------------------------------------------------------------*/
    /* Accept RFC-Connection                                            */
    /*------------------------------------------------------------------*/
#if defined(CRFCwithTHREADS)
	serverAppLock.Lock();
#endif
	rfc_handle = ::RfcAccept(new_argv);
#if defined(CRFCwithTHREADS)
	serverAppLock.Unlock();
#endif

	if (rfc_handle != RFC_HANDLE_NULL)
	{
		::RfcSetTrace(rfc_handle, globTraceLevel);  //  Set R/3 system trace level
		if (codepage !=  NULL)
			::RfcSetCodePage(rfc_handle, codepage);  //  Set codepage
	}

    sprintf(dbuf, "RfcAccept          handle = %d", rfc_handle);
    PRINT_RFC_CALL((int) lpParms, dbuf);

    if (rfc_handle == RFC_HANDLE_NULL)
    {
      rfc_error("RfcAccept", (int) lpParms);
      return 99;
    }

    rfc_thread[rfc_handle] = (int) lpParms;

    /*------------------------------------------------------------------*/
    /* Wait for RFC call                                                */
    /*------------------------------------------------------------------*/

    do
    {
      sprintf(dbuf, "Wait for next RFC call with RfcDispatch..........");
      PRINT_RFC_CALL((int) lpParms, dbuf);
      rfc_rc = ::RfcDispatch(rfc_handle);

    } while (rfc_rc == RFC_OK);

	if (rfc_rc == RFC_CLOSED)
	{
		::RfcLastErrorEx(&error_info);
		if (error_info.group == RFC_ERROR_CANCELLED)
			break;
	}
	else
		continue;
  }
  return 0;
}


#endif // defined(CRFCwithTHREADS)

